<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
class TMT_Heading extends Widget_Base {

	public function get_name(){
		return 'tmt-heading';
	}

	public function get_title(){
		return __( 'Heading', text_domain );
	}

	public function get_icon() {
		return 'eicon-t-letter-bold';
	}

	public function get_categories() {
		return [ text_domain ];
	}

	public function get_keywords() {
		return [ 'heading', 'title', 'text' ];
	}

	protected function register_controls() {
        $this->register_content_controls();
        $this->register_content_separator_controls();
	    $this->register_style_sub_heading_controls();
	    $this->register_style_before_heading_controls();
	    $this->register_style_after_heading_controls();
        $this->register_style_heading_controls();
		$this->register_style_separator_controls();
		$this->register_style_border_animation_controls();
	}



	protected function register_content_controls() {
		$this->start_controls_section(
			'content_section',
			[
				'label' => __( 'Content', text_domain )
			]
		);
		$this->add_responsive_control(
			'text_align',
			[
				'label' => __( 'Alignment', text_domain ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => __( 'Left', text_domain ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => __( 'Center', text_domain ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => __( 'Right', text_domain ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'default' => 'right',
				'toggle' => true,
				'selectors' => [
					'{{WRAPPER}} .heading_total, {{WRAPPER}} .separator,{{WRAPPER}} .tmt-heading' => 'justify-content: {{VALUE}};text-align:{{VALUE}}',
				],
			]
		);
        // heading //
		$this->add_control(
			'heading',
			[
				'label' => __( 'Heading', text_domain ),
				'type' => Controls_Manager::TEXTAREA,
				'separator' => 'before',
				'rows' => 2,
				'default' => __( 'Heading', text_domain ),
				'placeholder' => __( 'Type your text here', text_domain ),
			]
		);
		$this->add_control(
			'heading_tag',
			[
				'label' => __( 'Heading Tag', text_domain ),
				'type' => Controls_Manager::SELECT,
				'default' => 'h2',
				'options' => TMT_Title_Tags(),
				'condition' => [
					'heading!' => '',
				],
			]
		);
        // animation //
		$this->add_control(
			'show_border_animation',
			[
				'label' => __( 'Show Border Animation', text_domain ),
				'type' => Controls_Manager::SWITCHER,
				'separator' => 'before',
				'label_on' => __( 'Show', text_domain ),
				'label_off' => __( 'Hide', text_domain ),
				'return_value' => 'yes',
				'default!' => 'yes',
			]
		);
        // sub heading //
		$this->add_control(
			'show_sub_heading',
			[
				'label' => __( 'Show Sub Heading', text_domain ),
				'type' => Controls_Manager::SWITCHER,
				'separator' => 'before',
				'label_on' => __( 'Show', text_domain ),
				'label_off' => __( 'Hide', text_domain ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);
		$this->add_control(
			'sub_heading',
			[
				'type' => Controls_Manager::TEXTAREA,
				'rows' => 2,
				'placeholder' => __( 'Type your text here', text_domain ),
				'condition' => [
					'show_sub_heading' => 'yes'
				],
			]
		);
		$this->add_control(
			'sub_heading_tag',
			[
				'label' => __( 'Sub Heading Tag', text_domain ),
				'type' => Controls_Manager::SELECT,
				'default' => 'h3',
				'options' => TMT_Title_Tags(),
				'condition' => [
					'show_sub_heading' => 'yes'
				],
			]
		);
        // before heading //
		$this->add_control(
			'show_before_heading',
			[
				'label' => __( 'Show Before Heading', text_domain ),
				'type' => Controls_Manager::SWITCHER,
				'separator' => 'before',
				'label_on' => __( 'Show', text_domain ),
				'label_off' => __( 'Hide', text_domain ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);
		$this->add_control(
			'before_heading',
			[
				'label' => __( 'Before Heading', text_domain ),
				'type' => Controls_Manager::TEXTAREA,
				'rows' => 1,
				'placeholder' => __( 'Type your text here', text_domain ),
				'condition' => [
					'show_before_heading' => 'yes'
				],
			]
		);
		$this->add_control(
			'before_heading_tag',
			[
				'label' => __( 'Before Heading Tag', text_domain ),
				'type' => Controls_Manager::SELECT,
				'default' => 'h3',
				'options' => TMT_Title_Tags(),
				'condition' => [
					'show_before_heading' => 'yes'
				],
			]
		);
        // after heading //
		$this->add_control(
			'show_after_heading',
			[
				'label' => __( 'Show After Heading', text_domain ),
				'type' => Controls_Manager::SWITCHER,
				'separator' => 'before',
				'label_on' => __( 'Show', text_domain ),
				'label_off' => __( 'Hide', text_domain ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);
		$this->add_control(
			'after_heading',
			[
				'label' => __( 'After Heading', text_domain ),
				'type' => Controls_Manager::TEXTAREA,
				'rows' => 1,
				'placeholder' => __( 'Type your text here', text_domain ),
				'condition' => [
					'show_after_heading' => 'yes'
				],
			]
		);
		$this->add_control(
			'after_heading_tag',
			[
				'label' => __( 'After Heading Tag', text_domain ),
				'type' => Controls_Manager::SELECT,
				'default' => 'h3',
				'options' => TMT_Title_Tags(),
				'condition' => [
					'show_after_heading' => 'yes'
				],
			]
		);
        // link //
		$this->add_control(
			'heading_link',
			[
				'label' => __( 'Link', text_domain ),
				'type' => Controls_Manager::URL,
				'separator' => 'before',
				'placeholder' => __( 'https://www.themento.net/', text_domain ),
				'show_external' => true,
			]
		);
        // row & column //
		$this->add_control(
			'column_row',
			[
				'label' => __( 'Column/Row', text_domain ),
				'type' => Controls_Manager::SELECT,
				'default' => 'row',
				'options' => [
					'column'  => __( 'Column', text_domain ),
					'row' => __( 'Row', text_domain ),
				],
				'selectors' => [
					'{{WRAPPER}} .mix-headings' => 'flex-direction: {{VALUE}}',
				],
			]
		);
		$this->end_controls_section();
	}

	protected function register_content_separator_controls() {
		$this->start_controls_section(
			'content_separator_section',
			[
				'label' => __( 'Separator', text_domain ),
				'condition' => [
					'show_border_animation!' => 'yes',
				],

			]
		);
		$this->add_control(
			'separator_style',
			[
				'label' => __( 'Separator Style', text_domain ),
				'type' => Controls_Manager::SELECT,
				'default' => 'none',
				'options' => [
					'none'  => __( 'None', text_domain ),
					'line' => __( 'Line', text_domain ),
					'line_icon' => __( 'Line With Icon', text_domain ),
					'line_image' => __( 'Line With Image', text_domain ),
					'line_text' => __( 'Line With Text', text_domain ),
				],
			]
		);
		$this->add_control(
			'separator_position',
			[
				'label' => __( 'Separator Position', text_domain ),
				'type' => Controls_Manager::SELECT,
				'default' => 'between',
				'options' => [
					'between'  => __( 'Between Sub Heading & Heading', text_domain ),
					'top' => __( 'Top', text_domain ),
					'bottom' => __( 'Bottom', text_domain ),
					'before' => __( 'Before Heading', text_domain ),
					'after' => __( 'After Heading', text_domain ),
					'before_after' => __( 'Before & After Heading', text_domain ),
				],
				'condition' => [
					'separator_style!' => 'none',
				],
			]
		);
		$this->add_control(
			'separator_icon',
			[
				'label' => __( 'Icon', text_domain ),
				'type' => Controls_Manager::ICONS,
				'default' => [
					'value' => 'fas fa-star',
					'library' => 'solid',
				],
				'condition' => [
					'separator_style' => 'line_icon',
				],
			]
		);
		$this->add_control(
			'separator_image',
			[
				'label' => __( 'Choose Image', text_domain ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
				'condition' => [
					'separator_style' => 'line_image',
				],
			]
		);
		$this->add_control(
			'separator_text',
			[
				'label' => __( 'Title', text_domain ),
				'type' => Controls_Manager::TEXT,
				'default' => __( 'Text', text_domain ),
				'placeholder' => __( 'Type your text here', text_domain ),
				'condition' => [
					'separator_style' => 'line_text',
				],
			]
		);
		$this->end_controls_section();
	}

	protected function register_style_sub_heading_controls() {
		$this->start_controls_section(
			'content_style_sub_heading_section',
			[
				'label' => __( 'Sub Heading Style', text_domain ),
				'tab'   => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_sub_heading' => 'yes',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'sub_heading_typography',
				'label' => __( 'Typography', text_domain ),
				'selector' => '{{WRAPPER}} .sub-heading',
			]
		);
		$this->add_control(
			'sub_heading_color_style',
			[
				'label' => __( 'Color Style', text_domain ),
				'type' => Controls_Manager::SELECT,
				'default' => 'color',
				'options' => [
					'color'  => __( 'Color', text_domain ),
					'background' => __( 'Background', text_domain ),
				],
				'prefix_class' => 'tmt-sh-',
			]
		);
		$this->add_control(
			'sub_heading_color',
			[
				'label' => __( 'Color', text_domain ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .sub-heading' => 'color: {{VALUE}}',
				],
				'condition' => [
					'sub_heading_color_style' => 'color',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name' => 'sub_heading_background',
				'label' => __( 'Background', text_domain ),
				'types' => [ 'classic', 'gradient' ],
				'selector' => '{{WRAPPER}} .sub-heading',
				'condition' => [
					'sub_heading_color_style' => 'background',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			[
				'name' => 'sub_heading_text_shadow',
				'label' => __( 'Text Shadow', text_domain ),
				'selector' => '{{WRAPPER}} .sub-heading',
			]
		);
		$this->add_responsive_control(
			'sub_heading_margin',
			[
				'label' => __( 'Margin', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .sub-heading' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	protected function register_style_before_heading_controls() {
		$this->start_controls_section(
			'content_style_before_heading_section',
			[
				'label' => __( 'Before Heading Style', text_domain ),
				'tab'   => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_before_heading' => 'yes',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'before_heading_typography',
				'label' => __( 'Typography', text_domain ),
				'selector' => '{{WRAPPER}} .before_heading',
			]
		);
		$this->add_control(
			'before_heading_color',
			[
				'label' => __( 'Color', text_domain ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .before_heading' => 'color: {{VALUE}}',
				],
			]
		);
		$this->add_control(
			'before_heading_bg_color',
			[
				'label' => __( 'Background', text_domain ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .before_heading' => 'background: {{VALUE}}',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			[
				'name' => 'before_heading_text_shadow',
				'label' => __( 'Text Shadow', text_domain ),
				'selector' => '{{WRAPPER}} .before_heading',
			]
		);
		$this->add_responsive_control(
			'before_heading_padding',
			[
				'label' => __( 'Padding', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'allowed_diminsions' =>['right' , 'left'],
				'selectors' => [
					'{{WRAPPER}} .before_heading' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'before_heading_margin',
			[
				'label' => __( 'Margin', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'allowed_diminsions' =>['right' , 'left'],
				'selectors' => [
					'{{WRAPPER}} .before_heading' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'before_heading_radius',
			[
				'label' => __( 'Border Radius', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'allowed_diminsions' =>['right' , 'left'],
				'selectors' => [
					'{{WRAPPER}} .before_heading' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section();
	}

	protected function register_style_after_heading_controls() {
		$this->start_controls_section(
			'content_style_after_heading_section',
			[
				'label' => __( 'After Heading Style', text_domain ),
				'tab'   => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_after_heading' => 'yes',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'after_heading_typography',
				'label' => __( 'Typography', text_domain ),
				'selector' => '{{WRAPPER}} .after_heading',
			]
		);
		$this->add_control(
			'after_heading_color',
			[
				'label' => __( 'Color', text_domain ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .after_heading' => 'color: {{VALUE}}',
				],
			]
		);
		$this->add_control(
			'after_heading_bg_color',
			[
				'label' => __( 'Background', text_domain ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .after_heading' => 'background: {{VALUE}}',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			[
				'name' => 'after_heading_text_shadow',
				'label' => __( 'Text Shadow', text_domain ),
				'selector' => '{{WRAPPER}} .after_heading',
			]
		);
		$this->add_responsive_control(
			'after_heading_padding',
			[
				'label' => __( 'Padding', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'allowed_diminsions' =>['right' , 'left'],
				'selectors' => [
					'{{WRAPPER}} .after_heading' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'after_heading_margin',
			[
				'label' => __( 'Margin', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'allowed_diminsions' =>['right' , 'left'],
				'selectors' => [
					'{{WRAPPER}} .after_heading' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'after_heading_radius',
			[
				'label' => __( 'Border Radius', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'allowed_diminsions' =>['right' , 'left'],
				'selectors' => [
					'{{WRAPPER}} .after_heading' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section();
	}

	protected function register_style_heading_controls() {
		$this->start_controls_section(
			'content_style_heading_section',
			[
				'label' => __( 'Heading Style', text_domain ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'heading_color_style',
			[
				'label' => __( 'Color Style', text_domain ),
				'type' => Controls_Manager::SELECT,
				'default' => 'color',
				'options' => [
					'color'  => __( 'Color', text_domain ),
					'background' => __( 'Background', text_domain ),
				],
				'prefix_class' => 'tmt-h-',
			]
		);
		$this->add_control(
			'heading_color',
			[
				'label' => __( 'Color', text_domain ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .heading' => 'color: {{VALUE}}',
				],
				'condition' => [
					'heading_color_style' => 'color',
				],
			]
		);

		$this->add_control(
			'heading_bg_color',
			[
				'label' => __( 'Background', text_domain ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .heading' => 'background: {{VALUE}}',
				],
				'condition' => [
					'heading_color_style' => 'color',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name' => 'heading_background',
				'label' => __( 'Background', text_domain ),
				'types' => [ 'classic', 'gradient' ],
				'selector' => '{{WRAPPER}} .heading',
				'condition' => [
					'heading_color_style' => 'background',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'heading_typography',
				'label' => __( 'Typography', text_domain ),
				'selector' => '{{WRAPPER}} .heading',
			]
		);
		$this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			[
				'name' => 'heading_text_shadow',
				'label' => __( 'Text Shadow', text_domain ),
				'selector' => '{{WRAPPER}} .heading',
			]
		);
		$this->add_responsive_control(
			'space_heading_separator',
			[
				'label' => __( 'Space', text_domain ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
				],
				'default' => [
					'size' => 10,
				],
				'selectors' => [
					'{{WRAPPER}} .heading_total span' => 'padding: 0 {{SIZE}}px;',
				],
				'condition' => [
					'separator_style!' => 'none',
				],
			]
		);
		$this->add_responsive_control(
			'heading_padding',
			[
				'label' => __( 'Padding', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .heading' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'heading_margin',
			[
				'label' => __( 'Margin', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .heading' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'heading_radius',
			[
				'label' => __( 'Border Radius', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .heading' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section();
	}

	protected function register_style_separator_controls() {
		$this->start_controls_section(
			'content_separator_style_section',
			[
				'label' => __( 'Separator Style', text_domain ),
				'tab'   => Controls_Manager::TAB_STYLE,
				'condition' => [
					'separator_style!' => 'none',
					'show_border_animation!' => 'yes',
				],
			]
		);

		$this->start_controls_tabs('style_tabs');
		$this->start_controls_tab('style_line_tab', ['label' => __( 'Line', 'elementor' ),]);

		// option line //
		$this->add_control(
			'line_options',
			[
				'label' => __( 'Line Options', text_domain ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);
		$this->add_control(
			'line_style',
			[
				'label' => __( 'Line Style', text_domain ),
				'type' => Controls_Manager::SELECT,
				'default' => 'solid',
				'options' => [
					'solid'  => __( 'Solid', text_domain ),
					'dashed' => __( 'Dashed', text_domain ),
					'double' => __( 'Double', text_domain ),
					'dotted' => __( 'Dotted', text_domain ),
					'groove' => __( 'Groove', text_domain ),
				],
				'selectors' => [
					'{{WRAPPER}} .line' => 'border-style: {{VALUE}}',
				],
			]
		);
		$this->add_control(
			'separator_color',
			[
				'label' => __( 'Color', text_domain ),
				'type' => Controls_Manager::COLOR,
				'default' => '#000',
				'selectors' => [
					'{{WRAPPER}} .line' => 'border-color: {{VALUE}}',
				],
			]
		);
		$this->add_responsive_control(
			'separator_border_width',
			[
				'label' => __( 'Border Width', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
				],
				'default' => [
					'top' => '3',
					'right' => '20',
					'bottom' => '3',
					'left' => '20',
					'isLinked' => false,
				],
				'selectors' => [
					'{{WRAPPER}} .line ' => 'border-width: {{TOP}}px {{RIGHT}}px {{BOTTOM}}px {{LEFT}}px;',
				],
			]
		);
		$this->add_responsive_control(
			'separator_border_radius',
			[
				'label' => __( 'Border Radius', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .line,{{WRAPPER}} .separator' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'separator_margin',
			[
				'label' => __( 'Margin', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .separator' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'separator_padding',
			[
				'label' => __( 'Padding', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'default' => [
					'top' => '0',
					'right' => '10',
					'bottom' => '0',
					'left' => '10',
					'unit' => 'px',
					'isLinked' => false,
				],
				'selectors' => [
					'{{WRAPPER}} .separator i,{{WRAPPER}} .separator img,{{WRAPPER}} .separator span,{{WRAPPER}} .separator .line' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'separator_box_shadow',
				'label' => __( 'Box Shadow', text_domain ),
				'selector' => '{{WRAPPER}} .line',
			]
		);

        // Points_before_after_line //
		$this->add_control(
			'Points_before_after_line',
			[
				'label' => __( 'Points Before After Line', text_domain ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

        // Point_before_line //
		$this->add_control(
			'Point_before_line',
			[
				'label' => __( 'Point Before Line', text_domain ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => __( 'Show', text_domain ),
				'label_off' => __( 'Hide', text_domain ),
				'return_value' => 'yes',
				'default' => '',
			]
		);
		$this->add_control(
			'style_Point_before_line',
			[
				'label' => __( 'Line Style', text_domain ),
				'type' => Controls_Manager::SELECT,
				'default' => 'solid',
				'options' => [
					'solid'  => __( 'Solid', text_domain ),
					'dashed' => __( 'Dashed', text_domain ),
					'double' => __( 'Double', text_domain ),
					'dotted' => __( 'Dotted', text_domain ),
					'groove' => __( 'Groove', text_domain ),
				],
				'selectors' => [
					'{{WRAPPER}} .before_line' => 'border-style: {{VALUE}}',
				],
				'condition' => [
					'Point_before_line' => 'yes',
				],
			]
		);
		$this->add_control(
			'separator_color_Point_before_line',
			[
				'label' => __( 'Color', text_domain ),
				'type' => Controls_Manager::COLOR,
				'default' => '#000',
				'selectors' => [
					'{{WRAPPER}} .before_line' => 'border-color: {{VALUE}}',
				],
				'condition' => [
					'Point_before_line' => 'yes',
				],
			]
		);
		$this->add_responsive_control(
			'separator_border_width_Point_before_line',
			[
				'label' => __( 'Border Width', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .before_line' => 'border-width: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'Point_before_line' => 'yes',
				],
			]
		);
		
		$this->add_responsive_control(
			'separator_space_Point_before_line',
			[
				'label' => __( 'Space', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .before_line' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'Point_before_line' => 'yes',
				],
			]
		);
		
		$this->add_responsive_control(
			'separator_border_radius_Point_before_line',
			[
				'label' => __( 'Border Radius', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .before_line' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'Point_before_line' => 'yes',
				],
			]
		);

		// Point_after_line //
		$this->add_control(
			'Point_after_line',
			[
				'label' => __( 'Point After Line', text_domain ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => __( 'Show', text_domain ),
				'label_off' => __( 'Hide', text_domain ),
				'return_value' => 'yes',
				'default' => '',
			]
		);
		$this->add_control(
			'style_Point_after_line',
			[
				'label' => __( 'Line Style', text_domain ),
				'type' => Controls_Manager::SELECT,
				'default' => 'solid',
				'options' => [
					'solid'  => __( 'Solid', text_domain ),
					'dashed' => __( 'Dashed', text_domain ),
					'double' => __( 'Double', text_domain ),
					'dotted' => __( 'Dotted', text_domain ),
					'groove' => __( 'Groove', text_domain ),
				],
				'selectors' => [
					'{{WRAPPER}} .after_line' => 'border-style: {{VALUE}}',
				],
				'condition' => [
					'Point_after_line' => 'yes',
				],
			]
		);
		$this->add_control(
			'separator_color_Point_after_line',
			[
				'label' => __( 'Color', text_domain ),
				'type' => Controls_Manager::COLOR,
				'default' => '#000',
				'selectors' => [
					'{{WRAPPER}} .after_line' => 'border-color: {{VALUE}}',
				],
				'condition' => [
					'Point_after_line' => 'yes',
				],
			]
		);
		$this->add_responsive_control(
			'separator_border_width_Point_after_line',
			[
				'label' => __( 'Border Width', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .after_line' => 'border-width: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'Point_after_line' => 'yes',
				],
			]
		);
		
		$this->add_responsive_control(
			'separator_space_Point_after_line',
			[
				'label' => __( 'Space', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .after_line' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'Point_after_line' => 'yes',
				],
			]
		);
		
		$this->add_responsive_control(
			'separator_border_radius_Point_after_line',
			[
				'label' => __( 'Border Radius', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .after_line' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'Point_after_line' => 'yes',
				],
			]
		);
		$this->add_control(
			'flex-direction-hr-line',
			[
				'label' => __( 'Select Position Line', text_domain ),
				'type' => Controls_Manager::SELECT,
				'default' => 'row',
				'options' => [
					'column'  => __( 'Column', text_domain ),
					'row' => __( 'Row', text_domain ),
				],
				'selectors' => [
					'{{WRAPPER}} .hr-line' => 'flex-direction:{{VALUE}}',
				],
			]
		);

        // second_line //
		$this->add_control(
			'second_line',
			[
				'label' => __( 'Second Line', text_domain ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => __( 'Show', text_domain ),
				'label_off' => __( 'Hide', text_domain ),
				'return_value' => 'yes',
				'default' => 'no',
				'selectors' => [
					'{{WRAPPER}} .separator' => 'position: relative;',
					'{{WRAPPER}} .separator:before' => 'position: absolute;top: 0;content: "";display: block;',
				],
			]
		);
		$this->add_control(
			'second_separator_options',
			[
				'label' => __( 'Second Line Options', text_domain ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'second_line' => 'yes',
				],
			]
		);
		$this->add_control(
			'second_line_style',
			[
				'label' => __( 'Line Style', text_domain ),
				'type' => Controls_Manager::SELECT,
				'default' => 'solid',
				'options' => [
					'solid'  => __( 'Solid', text_domain ),
					'dashed' => __( 'Dashed', text_domain ),
					'double' => __( 'Double', text_domain ),
					'dotted' => __( 'Dotted', text_domain ),
					'groove' => __( 'Groove', text_domain ),
				],
				'selectors' => [
					'{{WRAPPER}} .separator:before' => 'border-style: {{VALUE}}',
				],
				'condition' => [
					'second_line' => 'yes',
				],
			]
		);
		$this->add_control(
			'second_separator_color',
			[
				'label' => __( 'Color', text_domain ),
				'type' => Controls_Manager::COLOR,
				'default' => '#000',
				'selectors' => [
					'{{WRAPPER}} .separator:before' => 'border-color: {{VALUE}}',
				],
				'condition' => [
					'second_line' => 'yes',
				],
			]
		);
		$this->add_responsive_control(
			'second_separator_height',
			[
				'label' => __( 'Height', text_domain ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
				],
				'default' => [
					'size' => 3,
				],
				'selectors' => [
					'{{WRAPPER}} .separator:before' => 'border-width: {{SIZE}}px;',
				],
				'condition' => [
					'second_line' => 'yes',
				],
			]
		);
		$this->add_responsive_control(
			'second_separator_width',
			[
				'label' => __( 'Width', text_domain ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 5,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => '%',
					'size' => 30,
				],
				'selectors' => [
					'{{WRAPPER}} .separator:before' => 'width: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'second_line' => 'yes',
				],
			]
		);
		$this->add_responsive_control(
			'second_separator_transform',
			[
				'label' => __( 'Position Y', text_domain ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => -100,
						'max' => 100,
					],
				],
				'default' => [
					'size' => -2,
				],
				'selectors' => [
					'{{WRAPPER}} .separator:before' => 'transform: translateY({{SIZE}}{{UNIT}});',
				],
				'condition' => [
					'second_line' => 'yes',
				],
			]
		);

		$this->end_controls_tab();
		$this->start_controls_tab('style_icon_text_img_tab',
		    [
			    'label' => __( 'Icon/Text/Image', 'elementor' ),
			    'condition' => [ 
				    'separator_style' => ['line_icon','line_image','line_text']
			    ]
			]
		);

        // icon & image & text //
		$this->add_control(
			'icon_options',
			[
				'label' => __( 'Icon Options', text_domain ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'separator_style' => 'line_icon',
				],
			]
		);
		$this->add_control(
			'text_options',
			[
				'label' => __( 'Text Options', text_domain ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'separator_style' => 'line_text',
				],
			]
		);
		$this->add_control(
			'icon_color',
			[
				'label' => __( 'Color', text_domain ),
				'type' => Controls_Manager::COLOR,
				'default' => '#000',
				'selectors' => [
					'{{WRAPPER}} .separator i,{{WRAPPER}} .separator span' => 'color: {{VALUE}}',
				],
				'condition' => [
					'separator_style' => ['line_text', 'line_icon'],
				],
			]
		);
		$this->add_responsive_control(
			'icon_size',
			[
				'label' => __( 'Size', text_domain ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
						'step' => 1,
					],
				],
				'default' => [
					'size' => 20,
				],
				'selectors' => [
					'{{WRAPPER}} .separator i,{{WRAPPER}} .separator span' => 'font-size: {{SIZE}}px;',
				],
				'condition' => [
					'separator_style' => ['line_text', 'line_icon'],
				],
			]
		);
		$this->add_responsive_control(
			'image_size',
			[
				'label' => __( 'Size', text_domain ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 5,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 100,
				],
				'selectors' => [
					'{{WRAPPER}} .separator img' => 'width: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'separator_style' => 'line_image',
				],
			]
		);

		$this->end_controls_tab();
		$this->end_controls_tabs();
		$this->end_controls_section();
	}

	protected function register_style_border_animation_controls() {
		$this->start_controls_section(
			'style_border_animation_section',
			[
				'label' => __( 'Border Animation', text_domain ),
				'tab'   => Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_border_animation'  => 'yes'
				],
			]
		);

		$this->add_control(
			'color_border_animation',
			[
				'label' => __( 'Color', text_domain ),
				'type' => Controls_Manager::COLOR,
				'default' => '#ffc009',
				'selectors' => [
					'{{WRAPPER}} .title-effect .bar' => 'background: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'opacity_border_animation',
			[
				'label' => __( 'Opacity', text_domain ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0.1,
						'max' => 0.9,
						'step' => 0.1,
					],
				],
				'default' => [
					'size' => 0.4,
				],
				'selectors' => [
					'{{WRAPPER}} .title-effect' => 'opacity: {{SIZE}};',
				],
			]
		);

		$this->add_control(
			'size_border_animation',
			[
				'label' => __( 'Size', text_domain ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
						'step' => 1,
					],
				],
				'default' => [
					'size' => 50,
				],
				'selectors' => [
					'{{WRAPPER}} .title-effect' => 'width: {{SIZE}}px; height:{{SIZE}}px',
				],
			]
		);

		$this->add_responsive_control(
			'position_border_animation',
			[
				'label' => __( 'Position', text_domain ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'allowed_dimensions' => ['top','right'],
                'default' => [
	                'top' => '-15',
				],
				'selectors' => [
					'{{WRAPPER}} .title-effect' => 'position:absolute; top:{{TOP}}{{UNIT}}; right:{{RIGHT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	protected function render() {
		$settings = $this->get_settings();
		$before_heading_tag = $settings['before_heading_tag'];
		$after_heading_tag = $settings['after_heading_tag'];
		$before_heading = $settings['before_heading'];
		$after_heading = $settings['after_heading'];
		// $text_align = $settings['text_align'];
		$sub_heading = $settings['sub_heading'];
		$heading = $settings['heading'];
		$sub_heading_tag = $settings['sub_heading_tag'];
		$heading_tag = $settings['heading_tag'];
		$separator_style = $settings['separator_style'];
		$show_border_animation = $settings['show_border_animation'];
		
		switch ($separator_style) {
			case 'none' :
				$line = "";
				break;
			case 'line' :
				$line = "<div class='separator hr-line flex align-items-center'><hr class='before_line'><hr class='line'><hr class='after_line'></div>";
				break;
			case 'line_icon' :
				$icon = $settings['separator_icon']['value'];
				$line = "<div class='separator flex align-items-center'><hr><i class='$icon'></i><hr class='line'></div>";
				break;
			case 'line_image' :
				$image = $settings['separator_image']['url'];
				$line = "<div class='separator flex align-items-center'><hr><img src='$image' alt='$image'><hr class='line'></div>";
				break;
			case 'line_text' :
				$separator_text = $settings['separator_text'];
				$line = "<div class='separator flex align-items-center'><hr><span>$separator_text</span><hr class='line'></div>";
				break;
		}
		$separator_position = $settings['separator_position'];
		// Link
		$heading_link = $settings['heading_link']['url'];
		if (!empty($heading_link) && !Plugin::$instance->editor->is_edit_mode()) {
			$target = $settings['heading_link']['is_external'] ? '_blank' : '_self';
			$this->add_render_attribute( 'heading', 'onclick', "window.open('$heading_link', '$target')" );
			$this->add_render_attribute( 'heading','class', 'pointer'  );
		}

		$this->add_render_attribute( 'heading','class', 'tmt-heading'  );

		$heading_render = $this->get_render_attribute_string( 'heading' );


		echo "<div $heading_render >";
            // line top //
			if ($separator_position == 'top') {echo $line;}

			// heading //
			echo "<div class='heading_total flex align-items-center'>";

			    // line before || before after //
				if ($separator_position == 'before' || $separator_position == 'before_after') {echo $line;}
				
				// border animation
                if($show_border_animation == 'yes'){
	                echo "<div class='title-effect'>"  
		                . "<div class='bar bar-top'></div>"
		                . "<div class='bar bar-right'></div>"
		                . "<div class='bar bar-bottom'></div>"
		                . "<div class='bar bar-left'></div>"
	                ."</div>";
	            }

			    // before heading     heading      after heading  //
			    echo "<span class='mix-headings flex flex-wrap'>";
			        // before heading //
				    if( $settings['show_before_heading'] == 'yes' ){echo "<$before_heading_tag class='before_heading'>$before_heading</$before_heading_tag>";}
				        // heading //
					    echo "<$heading_tag class='heading'>$heading</$heading_tag>";
			        // after heading //
		            if( $settings['show_after_heading'] == 'yes' ){echo "<$after_heading_tag class='after_heading'>$after_heading</$after_heading_tag>";}
			    echo "</span>";

                // line after || before after //
				if ($separator_position == 'after' || $separator_position == 'before_after') {echo $line;}
				
			echo "</div>";

            // line between heading & subheading //
			if ($separator_position == 'between') {echo $line;}

            // sub_heading //
			if ($settings['show_sub_heading']){echo "<$sub_heading_tag class='sub-heading'>$sub_heading</$sub_heading_tag>";}

            // line bottom //
			if ($separator_position == 'bottom') {echo $line;}
		echo "</div>";

	}
}
Plugin::instance()->widgets_manager->register( new TMT_Heading );